/** @file   gameanims.cpp
 * @brief   Implementation of GameAnims - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */
 
#include "gameanims.h"
#include "consts.h"
#include "weapon.h"
#include "www_assert.h"
#include <string>
#include <fstream>
#include <stdexcept>
#include "gfxid.h"
#include "eng2d.h"
#include "GfxManager.h"
#include "AnimId.h"
using namespace eng2d;
using std::vector;
using std::string;


namespace WeWantWar {

///
/// Constants, datatypes and static methods
/// ============================================================================

DataWarehouse* GameAnims::animationStore = 0;



/** Initilises the animations.
 */
int GameAnims::init()
{
  LOG_MESSAGE( "    GameAnims: Start reading animations..." );
  
  // Create the warehouse object.
  animationStore = new DataWarehouse;
  
  // Read the contents.
  try {
    FileList filelist( "data/anims/all_animations.txt" );
    WarehouseFactory::CreateWarehouse( animationStore, filelist );
    LOG_MESSAGE( "    GameAnims: Animations read." );
    
  } catch ( xRuntimeError except ) {
    delete animationStore;    animationStore = 0;
    string message = "        Reason: " + except.iMessage;
    LOG_MESSAGE( "        Animations failed." );
    LOG_MESSAGE( message.c_str() );
    
  }/* catch (...) {
    LOG_MESSAGE( "        XXX" );
    delete animationStore;
    animationStore = 0;
  }*/
  if ( animationStore == 0 ) {
    return -1;
  }
  
  // Assign the graphics to the animations.
  LOG_MESSAGE( "    GameAnims: Assign graphics..." );
  DataWarehouse::Iterator it = animationStore->Begin();
  while ( it != animationStore->End() ) {
    if ( it->second != 0 ) {
      AnimContainer* item = dynamic_cast<AnimContainer*>( it->second );
      int error = item->AssignGraphics( GfxManager::getWarehouse() );
      if ( error != KErrNone ) {
        string message = "        Reason: " + Int2String( error );
        string combineId = "        Combine id: " + Int2String( it->first );
        LOG_MESSAGE( "        GameAnims: Assigning failed!" );
        LOG_MESSAGE( message.c_str() );
        LOG_MESSAGE( combineId.c_str() );
        // Assigning graphics failed. Exit.
        delete animationStore;
        animationStore = 0;
        return -1;
      }
    }
    
    it++;
  }
  LOG_MESSAGE( "    GameAnims:: Assigning done" );
  return 0;
}



/** Destroyes the animations.
 */
void GameAnims::cleanup()
{
  delete animationStore;
  animationStore = 0;
}



/** Returns the player animations.
 */
const Animation& GameAnims::getAnimByWeapon( int aWeapon, int aAnimId )
{
  int combineUid = -1;
  switch ( aWeapon ) {
    case ( Weapon::W_CROWBAR ): {
      combineUid = AnimId::KPlayerCrowbar;
      break;
    }
    case ( Weapon::W_RIFLE ): {
      combineUid = AnimId::KPlayerRifle;
      break;
    }
    case ( Weapon::W_SHOTGUN ): {
      combineUid = AnimId::KPlayerRifle;
      break;
    }
    case ( Weapon::W_FLAMETHROWER ): {
      combineUid = AnimId::KPlayerRifle;
      break;
    }
    case ( Weapon::W_MINIGUN ): {
      combineUid = AnimId::KPlayerMinigun;
      break;
    }
    case ( Weapon::W_UZI ): {
      combineUid = AnimId::KPlayerUzi;
      break;
    }
    case ( Weapon::W_SNIPERRIFLE ): {
      combineUid = AnimId::KPlayerRifle;
      break;
    }
    case ( Weapon::W_ROCKETLAUNCHER ): {
      combineUid = AnimId::KPlayerLauncher;
      break;
    }
    default: {
      combineUid = AnimId::KPlayerRifle;
      break;
    }
  }
  AnimContainer* item = dynamic_cast<AnimContainer*>( animationStore->Find(combineUid) );
  return item->FindAnim( aAnimId );
}



/** Returns the requested AnimItem
 */
const AnimContainer* GameAnims::getAnimCombine( int aCombineUid )
{
  return dynamic_cast<AnimContainer*>( animationStore->Find(aCombineUid) );
}



const Animation& GameAnims::findAnimation( int aCombineUid, int aAnimIndex )
    throw ( Exception, eng2d::xOutOfBounds )
{
  const AnimContainer* animItem = getAnimCombine( aCombineUid );
  if ( animItem == 0 ) {
    string message = "Reference to null: GameAnims::findAnimation(" +
                     Int2String( aCombineUid) + ", " + Int2String( aAnimIndex) +
                     ")";
    throw Exception( message );
  }
  return animItem->FindAnim( aAnimIndex );
}



///
/// Constructors, destructor and operators
/// ============================================================================

/** Default constructor.
 */
GameAnims::GameAnims()
{
}



/** Destructor.
 */
GameAnims::~GameAnims()
{
}


} // end of namespace
